# -*- coding: utf-8 -*-
"""
Created on Sun Mar  5 13:23:21 2023

@author: Nicole
"""
import numpy as np
import matplotlib.pyplot as plt
import scipy.optimize as sciopt

""" Quelques constantes : 
"""
Ea = 80.e3          # énergie d'activation
k_0 = 1.e10         # facteur préexponentiel
R = 8.314           # constante des gaz parfaits

""" 
Rôle de T sur X_A pour une réaction d'ordre 1 non renversable dans un RPAC : 
On définit deux fonctions
- la première est la constante de vitesse (loi d'Arrhénius)
- la seconde donne le taux de conversion issu d'un bilan matière
"""

def arrhenius(T,Ea,k_0):
    return k_0*np.exp(-Ea/R/T)

def taux_conv(T,Ea,k_0,tau):        #tau est le temps de passage
    return tau*arrhenius(T,Ea,k_0)/(1 + tau*arrhenius(T,Ea,k_0))

T=np.linspace(200,600,500)  #Choix d'un intervalle de température

"""
Effets thermiques : marche adiabatique du RPAC
On définit une 3è fonction (endo) qui vient du bilan enthalpique
le coefficient ga est l'inverse de J, augmentation de température adiabatique
ga < 0 pour une réaction endothermique ; ga > 0 si exothermique.
"""

def endo(T,T_0,ga):
    return ga*(T-T_0)


"""
Point de fonctionnement par résolution numérique : cas endothermique
Pour une température T0 donnée, le point de fonctionnement est dans
l'intervalle [T0,T0+1/ga]
On utilise la méthode bisect (dichotomie) de scipy
"""

ga = -0.03
T0 = 450
T1 = 450 +1 /ga
tau = 10.

def racine(T,T0,ga,Ea,k0,tau):
    """
    Parameters
    ----------
    T : température en K.
    T0 : température à l'entrée du réacteur en K.
    ga : inverse de l'augmentation de température adiabatique.
    Ea : énergie d'activation en kJ/mol.
    k0 : facteur préexponentiel, en s^{-1}.
    tau : temps de passage en s.

    Returns
    -------
    la différence entre les taux de conversion calculés par le bilan de matière
    (cinétique d'ordre 1) et le bilan d'enthalpie.

    """
    return endo(T,T0,ga)-taux_conv(T, Ea, k_0, tau)

sol=sciopt.bisect(racine,T1,T0,args=(T0,ga,Ea,k_0,10))
print("température du point de fonctionnement (en K) =","{:.1f}".format(sol))


"""
Point de fonctionnement : cas exothermique
il peut y avoir plusieurs points de fonctionnement.
il faut donc trouver les intervalles dans lesquels faire la résolution
pour cela, on teste quand la fonction racine change de signe et on résout 
intervalle que l'on vient de détecter.
"""

"""
Choix de nouveaux paramètres avec lesquels il y a 3 points de fonctionnement
"""
ga = 5e-3
T0 = 340
Ea=70e3
k_0=2.e8
tau = 1
N=100
T=np.linspace(200,600,N)  #Choix d'un intervalle de température


# initialisation de la valeur courante de racine
a = racine(T0,T0,ga,Ea,k_0,1)
# initialisation de la liste des températures des points de fonctionnement
sol=[]
for i in range(1,N):   #on parcourt toutes les températures.
    if a*racine(T[i], T0, ga, Ea, k_0, 1) > 0 : # détecter un changement de signe
        a = racine(T[i], T0, ga, Ea, k_0, 1) #nouvelle valeur courante
    else :
        b = racine(T[i], T0, ga, Ea, k_0, 1) #
        res=sciopt.bisect(racine,T[i-1],T[i],args=(T0,ga,Ea,k_0,1))
        sol.append(round(res,1)) #on garde un seul chiffre après la virgule
        a = b #nouvelle valeur courante, après changement de signe

print("Températures des points de fonctionnement en K)",sol)

# Pour le fun, graphe de la différence entre les taux de conversion
plt.figure()
plt.plot(T,racine(T,T0,ga,Ea,k_0,1),color='black')
#plt.ylim([0,1])
plt.xlim([200,600])
plt.xlabel("$T$ (en K)")
plt.ylabel("$X_A$ cinétique -$X_A$ thermique")
plt.grid(which='both')
plt.show()